#pragma once
#include "graph.hpp"
#include <vector>
#include <algorithm>
#include <functional>

namespace atoms {

    // --- helper: check if a component contains a cycle ---
    inline bool component_has_cycle(const Graph& g, const std::vector<int>& comp) {
        std::vector<bool> vis(g.n, false);
        std::vector<int> in_comp(g.n, 0);
        for (int v : comp) in_comp[v] = 1;

        std::function<bool(int, int)> dfs = [&](int u, int parent) {
            vis[u] = true;
            for (auto e : g.adj[u]) {
                if (!in_comp[e.to]) continue;
                if (!vis[e.to]) {
                    if (dfs(e.to, u)) return true;
                }
                else if (e.to != parent) {
                    return true; // found cycle
                }
            }
            return false;
            };

        for (int v : comp) {
            if (!vis[v] && dfs(v, -1))
                return true;
        }
        return false;
    }

    // --- main: compute atoms of cyclic connectivity ---
    inline std::vector<std::vector<int>> atoms_of_cyclic_connectivity(
        const Graph& g,
        const std::vector<std::vector<int>>& cuts)
    {
        std::vector<std::vector<int>> parts;

        for (const auto& cut : cuts) {
            std::vector<bool> removed(g.edges.size(), false);
            for (int e : cut) removed[e] = true;

            std::vector<bool> vis(g.n, false);
            for (int i = 0; i < g.n; ++i) {
                if (!vis[i]) {
                    std::vector<int> comp;
                    std::vector<int> stack = { i };
                    vis[i] = true;

                    while (!stack.empty()) {
                        int v = stack.back(); stack.pop_back();
                        comp.push_back(v);
                        for (auto e : g.adj[v]) {
                            if (!removed[e.id] && !vis[e.to]) {
                                vis[e.to] = true;
                                stack.push_back(e.to);
                            }
                        }
                    }

                    if (component_has_cycle(g, comp))
                        parts.push_back(comp);
                }
            }
        }

        // --- minimal inclusion -> atoms ---
        std::vector<std::vector<int>> atoms;
        for (auto& A : parts) {
            bool minimal = true;
            for (auto& B : parts) {
                if (B.size() < A.size()) {
                    std::vector<int> sortedA = A;
                    std::vector<int> sortedB = B;
                    std::sort(sortedA.begin(), sortedA.end());
                    std::sort(sortedB.begin(), sortedB.end());
                    if (std::includes(sortedA.begin(), sortedA.end(),
                        sortedB.begin(), sortedB.end())) {
                        minimal = false;
                        break;
                    }
                }
            }
            if (minimal) atoms.push_back(A);
        }

        return atoms;
    }

} // namespace atoms